/*____________________________________________________________________________
	Copyright (C) 1997-1999 Networks Associates Technology, Inc.
	All rights reserved.
	
	This file contains miscellaneous utility functions needed for the PGPsdk.

	$Id: pgpUtilities.h,v 1.18 2001/02/12 07:47:59 wprice Exp $
____________________________________________________________________________*/

#ifndef Included_pgpUtilities_h	/* [ */
#define Included_pgpUtilities_h

#include <time.h>

#include "pgpPubTypes.h"
#include "pgpMemoryMgr.h"

#if PGP_MACINTOSH	/* [ */
#include <Files.h>
#endif	/* ] PGP_MACINTOSH */

#if PRAGMA_STRUCT_ALIGN
#pragma options align=mac68k	/* [ */
#endif

enum
{
	kPGPsdk20APIVersion		= 0x01000000,
	
	kPGPsdkAPIVersion		= kPGPsdk20APIVersion
};

#if PGP_MACINTOSH	/* [ */

#define kPGPMacFileCreator_PGPkeys			'pgpK'
#define kPGPMacFileCreator_PGPtools			'pgpM'
#define kPGPMacFileCreator_PGPnet			'PGPn'
#define kPGPMacFileCreator_PGPdisk			'pgpD'
#define kPGPMacFileCreator_PGPadmin			'PGPa'
#define kPGPMacFileCreator_DecryptedBinary	'????'
#define kPGPMacFileCreator_DecryptedText	'ttxt'

#define kPGPMacFileType_ArmorFile			'TEXT'
#define kPGPMacFileType_EncryptedData		'pgEF'
#define kPGPMacFileType_SignedData			'pgSF'
#define kPGPMacFileType_DetachedSig			'pgDS'
#define kPGPMacFileType_RandomSeed			'pgRS'
#define kPGPMacFileType_PrivRing			'pgRR'
#define kPGPMacFileType_PubRing				'pgPR'
#define kPGPMacFileType_Groups				'pgGR'
#define kPGPMacFileType_NetHosts			'pgHO'
#define kPGPMacFileType_NetRules			'pgRU'
#define kPGPMacFileType_Preferences			'pref'
#define kPGPMacFileType_DecryptedText		'TEXT'
#define kPGPMacFileType_DecryptedBinary		'BINA'
#define kPGPMacFileType_KeyShares			'pgSK'
#define kPGPMacFileType_Exported509Keys		'pgX5'

#endif	/* ] PGP_MACINTOSH */

PGP_BEGIN_C_DECLARATIONS

/*____________________________________________________________________________
	PGPsdk initialization

	Call PGPsdkXXXInit() before using that particular library.
	Call PGPsdkXXXCleanup() when you are done (but after disposing of any
		PGPContexts).
	
	You may call PGPsdkXXXInit multiple times (with no effect), but be sure
	to call the matching PGPsdkXXXCleanup() for each call to PGPsdkXXXInit().
____________________________________________________________________________*/
#define kPGPFlags_ForceLocalExecution	0x2
#define kPGPFlags_SuppressCacheThread	0x4

PGPError	PGPsdkInit( PGPFlags options );
PGPError	PGPsdkCleanup( void );

PGPError	PGPsdkNetworkLibInit( PGPFlags options );
PGPError	PGPsdkNetworkLibCleanup( void );

PGPError	PGPsdkUILibInit( PGPFlags options );
PGPError	PGPsdkUILibCleanup( void );

/*____________________________________________________________________________
	PGPsdk version information
	
	A version in the PGPsdk is expressed as follows:
	
	Major version (1 byte)
	Minor version (1 byte)
	Bug fix revision (1 byte)
	Reserved (1 byte)
	
	example: 1.7.1 = 0x01070100		
____________________________________________________________________________*/

PGPUInt32	PGPGetPGPsdkVersion( void );

/*____________________________________________________________________________
	PGPsdk version string
	
	Return a C string of the form:

	"PGPsdk 2.0 (C) Copyright 1999 Networks Associates Technology, Inc."
____________________________________________________________________________*/

PGPError	PGPGetPGPsdkVersionString( char versionString[ 256 ] );

/*____________________________________________________________________________
	PGPsdk API version information
	
	The API version of the PGPsdk is distinct from the externally visible
	version number. The API version is revised only when API changes are made,
	while the external version is changed whenever any code is changed, API or
	otherwise. The format of the API version is identical to the format of the
	external version. A change in the major API version implies incompatability
	with previous API versions while a change in the minor version implies API
	additions which maintain backwards compatability.
_____________________________________________________________________________*/

PGPUInt32	PGPGetPGPsdkAPIVersion( void );

/*____________________________________________________________________________
	PGPsdk context manipulation
_____________________________________________________________________________*/
	
typedef struct PGPCustomContextInfo
{
	PGPUInt32			sdkAPIVersion;	/* Use kPGPsdkAPIVersion */
	PGPMemoryMgrRef		memoryMgr;
	PGPUserValue		userValue;	

} PGPCustomContextInfo;


PGPError 	PGPNewContext( PGPUInt32 sdkAPIVersion, PGPContextRef *newContext );
PGPError 	PGPNewContextCustom( const PGPCustomContextInfo *contextInfo,
					PGPContextRef *newContext );

PGPError 	PGPFreeContext( PGPContextRef context );

PGPError 	PGPGetContextUserValue( PGPContextRef context,
					PGPUserValue *userValue );
PGPError 	PGPSetContextUserValue( PGPContextRef context,
					PGPUserValue userValue );

PGPError	PGPContextGetRandomBytes(PGPContextRef context,
					void *buf, PGPSize len );

PGPUInt32	PGPContextReserveRandomBytes(PGPContextRef context,
					PGPUInt32 minSize );

PGPMemoryMgrRef	PGPPeekContextMemoryMgr( PGPContextRef context );

/*____________________________________________________________________________
	PGP file management
	
	All files in PGP are represented using an opage data type PGPFileSpecRef.
	These data types are created using a fully qualified path or, on the
	Macintosh, an FSSpec. The 
____________________________________________________________________________*/

#if PGP_MACINTOSH	/* [ */

PGPError 	PGPNewFileSpecFromFSSpec( PGPContextRef context, const FSSpec *spec,
					PGPFileSpecRef *ref );
PGPError 	PGPGetFSSpecFromFileSpec( PGPFileSpecRef fileRef, FSSpec *spec );

#else	/* ][ */

PGPError 	PGPNewFileSpecFromFullPath( PGPContextRef context,
					const char *path, PGPFileSpecRef *ref );
/*
** The full path is an allocated object which needs to be deallocated with
** PGPFreeData()
*/

PGPError 	PGPGetFullPathFromFileSpec( PGPFileSpecRef fileRef,
					char **fullPathPtr);

#endif	/* ] */

PGPError 	PGPCopyFileSpec( PGPFileSpecRef fileRef, PGPFileSpecRef *ref );
PGPError 	PGPFreeFileSpec( PGPFileSpecRef fileRef );

PGPError	PGPRenameFile( PGPFileSpecRef fileRef, const char *newName );
PGPError	PGPDeleteFile( PGPFileSpecRef fileRef );
	
/*____________________________________________________________________________
	PGP time/date functions
____________________________________________________________________________*/

PGPTime 	PGPGetTime(void);

/* these use time_t type as returned by time() in Std C libraries */
time_t 		PGPGetStdTimeFromPGPTime( PGPTime theTime );
PGPTime 	PGPGetPGPTimeFromStdTime( time_t theTime );

/* year, month, day may be NULL if desired */
void 		PGPGetYMDFromPGPTime( PGPTime theTime, PGPUInt16 *year,
					PGPUInt16 *month, PGPUInt16 *day );

#if PGP_MACINTOSH	/* [ */

PGPUInt32 	PGPTimeToMacTime( PGPTime theTime );
PGPTime 	PGPTimeFromMacTime( PGPUInt32 theTime );

#endif	/* ] PGP_MACINTOSH */

/*____________________________________________________________________________
	MacBinary support

	Examine the input file to see if it's a MacBinary file.  If it is
	not a MacBinary file, then the original file is unaltered.
	Otherwise, the file is converted and the original file is deleted.
	
	The resulting file is designated by 'outPGPSpec' and may have a different
	name than the original.
	
	If the file is a TEXT file, appropriate line-end conversion is done.
	
	creator and type code pointers may be
	null but otherwise contain the mac creator and type.
	
	This routine can be called on a Mac, but generally doesn't need to be.
____________________________________________________________________________*/

PGPError	PGPMacBinaryToLocal( PGPFileSpecRef inSpec,
					PGPFileSpecRef * outSpec, PGPUInt32 * macCreator,
					PGPUInt32 * macTypeCode );

/*____________________________________________________________________________
	Notification callbacks on PGPKeyDB changes from SDK Service
_____________________________________________________________________________*/
	
typedef void (*PGPNotificationHandlerProc)( PGPUserValue userValue );

PGPError	PGPSetNotificationCallback( PGPNotificationHandlerProc proc,
					PGPUserValue userValue );

/*____________________________________________________________________________
	FIPS 140-1 Support
____________________________________________________________________________*/

enum PGPsdkSelfTest_
{
	kPGPsdkSelfTest_Invalid		= 0,
	kPGPsdkSelfTest_FirstTest	= 1,
	
	kPGPsdkSelfTest_3DES		= 1,
	kPGPsdkSelfTest_DSA			= 2,
	kPGPsdkSelfTest_Integrity	= 3,
	
	kPGPsdkSelfTest_LastTest	= kPGPsdkSelfTest_Integrity,
	
	PGP_ENUM_FORCE( PGPsdkSelfTest_ )
};

PGPENUM_TYPEDEF( PGPsdkSelfTest_, PGPsdkSelfTest );

PGPError	PGPEnableFIPSMode(void);
PGPError	PGPGetSDKErrorState(void);
PGPError	PGPResetSDKErrorState(void);
PGPError	PGPRunSDKSelfTest(PGPsdkSelfTest whichTest);
PGPError	PGPRunAllSDKSelfTests(void);

PGP_END_C_DECLARATIONS

#if PRAGMA_STRUCT_ALIGN
#pragma options align=reset			/* ] */
#endif

#endif /* ] Included_pgpUtilities_h */

/*__Editor_settings____

	Local Variables:
	tab-width: 4
	End:
	vi: ts=4 sw=4
	vim: si
_____________________*/
